# app.py
import os, time, requests, shutil, base64
from flask import Flask, render_template, request, redirect, url_for, flash
from werkzeug.utils import secure_filename
import pandas as pd
import numpy as np
import cv2
import tensorflow as tf
from train_model import train_gauge_model  # import your training function

# ---------------- CONFIG ----------------
UPLOAD_FOLDER = "uploads"
CSV_FILE = "readings.csv"
MODEL_FILE = "gauge_reader_model.h5"
IMG_SIZE = 128
ALLOWED_EXT = {"png", "jpg", "jpeg"}
MAX_CONTENT_LENGTH = 6 * 1024 * 1024   # 6 MB max upload
MODEL_URL = os.getenv("MODEL_URL")  # Optional: download model from remote

# ✅ Set defaults
USE_GEMINI_FOR_PREDICT = False   # Predict & Save → Local model
USE_GEMINI_FOR_CHECK = True      # Check Reading → Gemini API

# ✅ Your Gemini API key (replace if needed)
GEMINI_API_KEY = "AIzaSyBK9FR_W1O7Pv7beerftBQmXJdQ27CG81c"

# ✅ Updated Gemini endpoint (working version)
GEMINI_URL = (
    "https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-pro-001:generateContent?key="
    + GEMINI_API_KEY
)

os.makedirs(UPLOAD_FOLDER, exist_ok=True)

# ---------------- MODEL LOADING ----------------
if MODEL_URL and not os.path.exists(MODEL_FILE):
    print("Downloading model from MODEL_URL...")
    r = requests.get(MODEL_URL, stream=True, timeout=60)
    r.raise_for_status()
    with open(MODEL_FILE, "wb") as f:
        shutil.copyfileobj(r.raw, f)
    print("Model downloaded.")

if not os.path.exists(MODEL_FILE):
    raise FileNotFoundError(f"{MODEL_FILE} not found. Upload it or set MODEL_URL.")

model = tf.keras.models.load_model(MODEL_FILE, compile=False)

# ---------------- APP INIT ----------------
app = Flask(__name__)
app.config["UPLOAD_FOLDER"] = UPLOAD_FOLDER
app.config["MAX_CONTENT_LENGTH"] = MAX_CONTENT_LENGTH
app.secret_key = os.getenv("FLASK_SECRET", "replace-this-secret")


# ---------------- HELPERS ----------------
def allowed_file(filename):
    return "." in filename and filename.rsplit(".", 1)[1].lower() in ALLOWED_EXT


def preprocess_image(path):
    img = cv2.imread(path)
    if img is None:
        raise ValueError("Unable to read image")
    img = cv2.resize(img, (IMG_SIZE, IMG_SIZE))
    img = img / 255.0
    return np.expand_dims(img.astype("float32"), axis=0)


def predict_with_model(local_path):
    """Predict using local TensorFlow model."""
    arr = preprocess_image(local_path)
    pred = model.predict(arr, verbose=0)[0][0]
    return float(pred)


def predict_with_gemini(image_path):
    """Predict using Google Gemini Vision API."""
    with open(image_path, "rb") as f:
        img_data = f.read()

    img_b64 = base64.b64encode(img_data).decode("utf-8")
    prompt = (
        "Read the pressure gauge shown in this image and return only the numeric value. "
        "Do not include text, symbols, or units — only the number."
    )

    payload = {
        "contents": [
            {
                "parts": [
                    {"text": prompt},
                    {"inline_data": {"mime_type": "image/jpeg", "data": img_b64}},
                ]
            }
        ]
    }

    try:
        response = requests.post(GEMINI_URL, json=payload, timeout=60)
        response.raise_for_status()
        data = response.json()
        text = data["candidates"][0]["content"]["parts"][0]["text"].strip()
        return float(text)
    except Exception as e:
        raise ValueError(f"Gemini API error: {e}")


def predict_reading(local_path):
    """Use only TensorFlow model for Predict & Save."""
    if USE_GEMINI_FOR_PREDICT:
        print("🔮 Using Gemini Vision API for prediction...")
        return predict_with_gemini(local_path)
    else:
        print("🧠 Using local TensorFlow model for prediction...")
        return predict_with_model(local_path)


def append_row(filename, reading, reporting_id):
    """Append a new reading row to CSV."""
    if os.path.exists(CSV_FILE):
        df = pd.read_csv(CSV_FILE)
        next_id = int(df["id"].max()) + 1 if ("id" in df.columns and not df.empty) else 1
    else:
        df = pd.DataFrame(columns=["id", "filename", "reading", "reporting_id"])
        next_id = 1
    new_row = {
        "id": next_id,
        "filename": filename,
        "reading": reading,
        "reporting_id": reporting_id,
    }
    df = pd.concat([df, pd.DataFrame([new_row])], ignore_index=True)
    df.to_csv(CSV_FILE, index=False)
    return next_id


# ---------------- ROUTES ----------------
@app.route("/", methods=["GET", "POST"])
def index():
    last_entry = None

    if request.method == "POST":
        if "image" not in request.files:
            flash("No file part")
            return redirect(request.url)
        file = request.files["image"]
        reporting_id = request.form.get("reporting_id", "").strip()
        if not reporting_id:
            flash("Please enter a Reporting ID")
            return redirect(request.url)
        if file.filename == "":
            flash("No selected file")
            return redirect(request.url)
        if file and allowed_file(file.filename):
            filename = secure_filename(file.filename)
            base, ext = os.path.splitext(filename)
            safe_name = f"{base}_{int(time.time()*1000)}{ext}"
            local_path = os.path.join(app.config["UPLOAD_FOLDER"], safe_name)
            file.save(local_path)

            try:
                # ✅ Now only TensorFlow model is used
                reading = predict_reading(local_path)
            except Exception as e:
                flash(f"Prediction error: {e}")
                return redirect(request.url)

            new_id = append_row(safe_name, reading, reporting_id)

            last_entry = {
                "id": new_id,
                "filename": safe_name,
                "reading": reading,
                "reporting_id": reporting_id,
            }

            return render_template(
                "result.html",
                id=new_id,
                filename=safe_name,
                predicted=reading,
                final_reading=reading,
                reporting_id=reporting_id,
                last_entry=last_entry,
            )
        else:
            flash("Allowed image types: png, jpg, jpeg")
            return redirect(request.url)

    # Show last entry if CSV exists
    if os.path.exists(CSV_FILE):
        df = pd.read_csv(CSV_FILE)
        if not df.empty:
            last_row = df.iloc[-1]
            last_entry = {
                "id": last_row["id"],
                "filename": last_row["filename"],
                "reading": last_row["reading"],
                "reporting_id": last_row["reporting_id"],
            }

    return render_template("index.html", last_entry=last_entry)


# ✅ Check Reading — Compare TensorFlow vs Gemini
@app.route("/correct", methods=["POST"])
def correct():
    reporting_id = request.form.get("reporting_id")

    if not reporting_id:
        flash("Missing Reporting ID")
        return redirect(url_for("index"))

    if not os.path.exists(CSV_FILE):
        flash("CSV file not found")
        return redirect(url_for("index"))

    try:
        df = pd.read_csv(CSV_FILE)
        if df.empty:
            flash("CSV file is empty")
            return redirect(url_for("index"))

        mask = df["reporting_id"].astype(str) == str(reporting_id)
        if not mask.any():
            flash(f"Reporting ID {reporting_id} not found in CSV")
            return redirect(url_for("index"))

        # Get the entry
        row = df.loc[mask].iloc[0]
        filename = row["filename"]
        local_path = os.path.join(app.config["UPLOAD_FOLDER"], filename)

        if not os.path.exists(local_path):
            flash(f"Image for Reporting ID {reporting_id} not found on disk.")
            return redirect(url_for("index"))

        # Step 1: Predict using TensorFlow model
        model_reading = predict_with_model(local_path)

        # Step 2: Predict using Gemini (only for check)
        gemini_reading = None
        try:
            if USE_GEMINI_FOR_CHECK:
                gemini_reading = predict_with_gemini(local_path)
        except Exception as e:
            flash(f"Gemini API failed: {e}")
            return redirect(url_for("index"))

        # Step 3: Compare and update
        if gemini_reading is not None and abs(model_reading - gemini_reading) > 0.5:
            df.loc[mask, "reading"] = gemini_reading
            df.to_csv(CSV_FILE, index=False)
            flash(
                f"🔄 Reading updated using Gemini. Old: {model_reading:.2f}, New: {gemini_reading:.2f}"
            )
        else:
            flash(
                f"✅ Gemini and Model readings match closely ({model_reading:.2f}). No change made."
            )

    except Exception as e:
        flash(f"Error during correction: {e}")

    return redirect(url_for("index"))


# ✅ Training Route
@app.route("/train", methods=["GET", "POST"])
def train():
    if request.method == "POST":
        try:
            msg = train_gauge_model()
            flash(msg)
        except Exception as e:
            flash(f"❌ Training failed: {e}")
        return redirect(url_for("train"))
    return render_template("train.html")


# ---------------- MAIN ----------------
if __name__ == "__main__":
    app.run(host="0.0.0.0", port=int(os.getenv("PORT", 5000)))
